package Pod::Abstract::Filter::ptree;
use strict;
use warnings;

use base qw(Pod::Abstract::Filter);
use Pod::Abstract::BuildNode qw(node);

our $VERSION = '0.26';

=head1 NAME

Pod::Abstract::Filter::ptree - convert the incoming document to a
summarised parse tree, and dump that into a verbatim block.

=head1 DESCRIPTION

This can be very useful to understand the generated structure of a POD
file when you are building your own filters or code based on
L<Pod::Abstract>.

=head1 USAGE

 $ paf ptree bin/paf
 
 ...

 Parse Tree
       1 [[ROOT]]
       2   [#cut] #!/usr/bin/perl
       3   [#cut] package paf;use strict;use warnings;
       4   [#cut] use Pod::Abstract;use Pod::Abstract::Filter;
       5   [#cut] use File::Temp qw(tempfile tempdir);
       8   [head1] NAME
       9     [:paragraph]
      10       [:text] paf - Pod Abstract Filter. Transform Pod documents from t
 <... etc>

=head1 METHODS

=head2 filter

This is the only method for the module, and it just makes use of the
L<Pod::Abstract::Node/ptree> method to generate a visual parse tree,
and nests that into a heading generated by L<Pod::Abstract::BuildNode>.

=cut

sub filter {
    my $self = shift;
    my $pa = shift;

    my $ptree = node->verbatim( $pa->ptree );
    my $pt_block = node->head1('Parse Tree');
    $pt_block->nest($ptree);
    
    $pt_block->coalesce_body(':text');
    
    return $pt_block;
}

1;